import pandas as pd
import numpy as np
import ipywidgets as widgets
import plotly as py
import plotly.graph_objs as go
from plotly.subplots import make_subplots



def techid_dropdowns(df):
	"""
	input condensed loadshape with metadata (cdls_md), outputs 4 jupyter widgets dropdowns (TechID, BldgType, Peak/Wkday/Wkend) based on the data provided

	:df: Pandas dataframe of condensed hourly load shapes (Peak, Avg, End), with merged meta data (TechID, BldgType, BldgVint, BldgLoc, ...)
	"""
	
	techID_list = list(df["TechID"].unique())
	
	#cz_list = []
	#cz_list.append("All")
	
	style = {'description_width': 'initial'}
	shape_choice = {"peak day kWh": 0, "average weekday kWh": 1, "average weekend kWh":2}
	dd_techID = widgets.Dropdown(options=techID_list, description="Choose TechID", style=style, layout={'width': 'max-content'})
	
	#dd_cz = widgets.Dropdown(options = cz_list, description="Choose CZ", style=style)
	dd_shape_picker = widgets.Dropdown(options = shape_choice, 
										 description="Choose peak/wkday/wkend to view (set version)", 
										 style=style,
										layout={'width': 'max-content'})
	
	return dd_techID, dd_shape_picker

def version_dropdowns(df):
	'''
	'''
	ver_list = list(df["Version"].unique())
	style = {'description_width': 'initial'}
	dd_version = widgets.Dropdown(options=ver_list, description="Choose DEER version", style=style, layout={'width':'max-content'})

	return dd_version

def bldgtype_dropdown(df):
	'''
	:df: Pandas dataframe of filtered df based on techID

	'''
	bldgtype_def = {"Asm" : "Assembly",
	"EPr" : "Education - Primary School",
	"ESe" : "Education - Secondary School",
	"ECC" : "Education - Community College",
	"EUn" : "Education - University",
	"Gro" : "Grocery",
	"Hsp" : "Health/Medical - Hospital",
	"Nrs" : "Health/Medical - Nursing Home",
	"Htl" : "Lodging - Hotel",
	"Mtl" : "Lodging - Motel",
	"MBT" : "Manufacturing - Bio/Tech",
	"MLI" : "Manufacturing - Light Industrial",
	"OfL" : "Office - Large",
	"OfS" : "Office - Small",
	"RSD" : "Restaurant - Sit-Down",
	"RFF" : "Restaurant - Fast-Food",
	"Rt3" : "Retail - Multistory Large",
	"RtL" : "Retail - Single-Story Large",
	"RtS" : "Retail - Small",
	"SCn" : "Storage - Conditioned",
	"SUn" : "Storage - Unconditioned",
	"WRf" : "Warehouse - Refrigerated",
	"SFm" : "Residential - Single Family",
	"MFm" : "Residential - Multifamily",
	"SMo" : "Residential - Single-Wide Mobile",
	"DMo" : "Residential - Double-Wide Mobile",
	"ERC" : "Education - Relocatable Classroom",
	"EUD" : "Education - University Dormitory",
	"HGR" : "Hotel Guest Room"
	}

	bldgtype_list = list(df["BldgType"].unique()) #this is bldgtype list of filtered df by techID in previous function
	

	if "Any" in list(df["BldgType"].unique()):
		bldg_type = ['Any']
	else:
		bldg_type = []
		for item in bldgtype_list:
			bldg_type.append(bldgtype_def[item])


	style = {'description_width': 'initial'}
	dd_bldgtype = widgets.Dropdown(options = bldg_type, description="Choose building type", style=style)

	return dd_bldgtype




def datafilter(lsdf, ratiosdf, techID_selection, version_selection, bldgtype_selection):
	"""
	Returns filtered datasets (cdls and ratios) according to filter selection.

	:lsdf: cdls loadshape data with metadata (Pandas dataframe)
	:ratiosdf: ratios loadshape data with metadata (Pandas dataframe)
	:techID_selection: techID (string) (modified for Dash version)
	:bldgtype_selection: building type (string) (modified for Dash version)
	:version_selection: the DEER version (20xx) (string) (for Dash version)
	
	"""
	bldgtype_def_rev = {"Assembly": "Asm",
	"Education - Primary School": "EPr",
	"Education - Secondary School": "ESe",
	"Education - Community College": "ECC",
	"Education - University": "EUn",
	"Grocery": "Gro",
	"Health/Medical - Hospital": "Hsp",
	"Health/Medical - Nursing Home": "Nrs",
	"Lodging - Hotel": "Htl",
	"Lodging - Motel": "Mtl",
	"Manufacturing - Bio/Tech": "MBT",
	"Manufacturing - Light Industrial": "MLI",
	"Office - Large": "OfL",
	"Office - Small": "OfS",
	"Restaurant - Sit-Down": "RSD",
	"Restaurant - Fast-Food": "RFF",
	"Retail - Multistory Large": "Rt3",
	"Retail - Single-Story Large": "RtL",
	"Retail - Small": "RtS",
	"Storage - Conditioned": "SCn",
	"Storage - Unconditioned": "SUn",
	"Warehouse - Refrigerated": "WRf",
	"Residential - Single Family": "SFm",
	"Residential - Multifamily": "MFm",
	"Residential - Single-Wide Mobile": "SMo",
	"Residential - Double-Wide Mobile": "DMo",
	"Education - Relocatable Classroom" : "ERC",
	"Education - University Dormitory" : "EUD",
	"Hotel Guest Room" : "HGR"}

	cond0 = lsdf["Version"] == version_selection
	cond1 = lsdf["TechID"] == techID_selection


	if bldgtype_selection == "Any":
		cond2 = lsdf["BldgType"] == "Any"
	else:
		cond2 = lsdf["BldgType"] == bldgtype_selection
		
	cond0_r = ratiosdf["Version"] == version_selection
	cond1_r = ratiosdf["TechID"]==techID_selection

	if bldgtype_selection == "Any":
		cond2_r = ratiosdf["BldgType"] == "Any"
	else:
		cond2_r = ratiosdf["BldgType"]==bldgtype_selection

	df_filtered = lsdf[cond0 & cond1 & cond2]
	ratios_filtered = ratiosdf[cond0_r & cond1_r & cond2_r]
	
	return df_filtered, ratios_filtered



def breakDownSeasonalTraces_py(df, version):
	"""
	This function is an internal function meant to be used in other functions here.
	It breaks down filtered condensed load shapes from a single CZ into 12 set of monthly plotting traces (Plotly traces objects), for that CZ
	The Plotly traces object are used to produce the corresponding plots later.

	:df: condensed, filtered cdls to a single CZ or BldgLoc in data (Pandas dataframe)
	:version: any string value to distinguish this set of load shapes from other sets (string)
	"""
	ls_name = list(df['BldgLoc'].unique())[0]
	jan_df = df[df["Month"]==1][["Month","ID", "Hour", "Peak","Avg","End"]] 
	feb_df = df[df["Month"]==2][["Month","ID", "Hour", "Peak","Avg","End"]] 
	mar_df = df[df["Month"]==3][["Month","ID", "Hour", "Peak","Avg","End"]] 
	apr_df = df[df["Month"]==4][["Month","ID", "Hour", "Peak","Avg","End"]] 
	may_df = df[df["Month"]==5][["Month","ID", "Hour", "Peak","Avg","End"]] 
	jun_df = df[df["Month"]==6][["Month","ID", "Hour", "Peak","Avg","End"]] 
	jul_df = df[df["Month"]==7][["Month","ID", "Hour", "Peak","Avg","End"]] 
	aug_df = df[df["Month"]==8][["Month","ID", "Hour", "Peak","Avg","End"]] 
	sep_df = df[df["Month"]==9][["Month","ID", "Hour", "Peak","Avg","End"]] 
	oct_df = df[df["Month"]==10][["Month","ID", "Hour", "Peak","Avg","End"]] 
	nov_df = df[df["Month"]==11][["Month","ID", "Hour", "Peak","Avg","End"]] 
	dec_df = df[df["Month"]==12][["Month","ID", "Hour", "Peak","Avg","End"]]
	
	jan_peak = go.Scatter(x=jan_df["Hour"], y = jan_df["Peak"], mode="lines+markers",name=f"January peak, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	jan_avg = go.Scatter(x=jan_df["Hour"], y = jan_df["Avg"], mode="lines+markers",name=f"January wkday, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	jan_end = go.Scatter(x=jan_df["Hour"], y = jan_df["End"], mode="lines+markers",name=f"January wkend, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	
	apr_peak = go.Scatter(x=apr_df["Hour"], y = apr_df["Peak"], mode="lines+markers",name=f"April peak, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	apr_avg = go.Scatter(x=apr_df["Hour"], y = apr_df["Avg"], mode="lines+markers",name=f"April wkday, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	apr_end = go.Scatter(x=apr_df["Hour"], y = apr_df["End"], mode="lines+markers",name=f"April wkend, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))

	jul_peak = go.Scatter(x=jul_df["Hour"], y = jul_df["Peak"], mode="lines+markers",name=f"July peak, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	jul_avg = go.Scatter(x=jul_df["Hour"], y = jul_df["Avg"], mode="lines+markers",name=f"July wkday, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	jul_end = go.Scatter(x=jul_df["Hour"], y = jul_df["End"], mode="lines+markers",name=f"July wkend, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))

	oct_peak = go.Scatter(x=oct_df["Hour"], y = oct_df["Peak"], mode="lines+markers",name=f"October peak, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	oct_avg = go.Scatter(x=oct_df["Hour"], y = oct_df["Avg"], mode="lines+markers",name=f"October wkday, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	oct_end = go.Scatter(x=oct_df["Hour"], y = oct_df["End"], mode="lines+markers",name=f"October wkend, {ls_name}, {version}", line=dict(shape='spline'), hoverlabel = dict(namelength=-1))
	
	season_traces = [[jan_peak,jan_avg,jan_end],\
					 [apr_peak,apr_avg,apr_end],\
					 [jul_peak,jul_avg,jul_end],\
					 [oct_peak,oct_avg,oct_end]]

	return ls_name, season_traces



#functions to deal with monthly factor traces
def assembleMonthlyFactor_py(df):
	"""
	This function is an internal function meant to be used in other functions here.
	It returns split-up dataframes of condensed load shape monthly factors for the 16 climate zones

	:df: filtered ratios dataframe (Pandas dataframe)
	"""
	mf_cz1 = df[df["BldgLoc"]=="CZ01"]
	mf_cz2 = df[df["BldgLoc"]=="CZ02"]
	mf_cz3 = df[df["BldgLoc"]=="CZ03"]
	mf_cz4 = df[df["BldgLoc"]=="CZ04"]
	mf_cz5 = df[df["BldgLoc"]=="CZ05"]
	mf_cz6 = df[df["BldgLoc"]=="CZ06"]
	mf_cz7 = df[df["BldgLoc"]=="CZ07"]
	mf_cz8 = df[df["BldgLoc"]=="CZ08"]
	mf_cz9 = df[df["BldgLoc"]=="CZ09"]
	mf_cz10 = df[df["BldgLoc"]=="CZ10"]
	mf_cz11 = df[df["BldgLoc"]=="CZ11"]
	mf_cz12 = df[df["BldgLoc"]=="CZ12"]
	mf_cz13 = df[df["BldgLoc"]=="CZ13"]
	mf_cz14 = df[df["BldgLoc"]=="CZ14"]
	mf_cz15 = df[df["BldgLoc"]=="CZ15"]
	mf_cz16 = df[df["BldgLoc"]=="CZ16"]
	mf = [mf_cz1,
		  mf_cz2,
		  mf_cz3,
		  mf_cz4,
		  mf_cz5,
		  mf_cz6,
		  mf_cz7,
		  mf_cz8,
		  mf_cz9,
		  mf_cz10,
		  mf_cz11,
		  mf_cz12,
		  mf_cz13,
		  mf_cz14,
		  mf_cz15,
		  mf_cz16]
	return mf

def monthlyTraces_py(df_mf, ls_name, version):
	"""
	This function is an internal function meant to be used in other functions here.
	prepare graphing traces for monthly factors - per CZ, need to include ls name

	:df_mf: each individual dataframes in the mf dataset, each should cover 1 CZ (Pandas dataframe) 
	:ls_name: load shape name produced using breakDownSeasonalTraces_py() (string)
	:version: any string value to distinguish this set of load shapes from other sets (string)
	"""
	monthly_load_pct = go.Bar(x=df_mf["Month"], y=df_mf["Mo Brk"], name=f"Monthly % Load, {ls_name}, {version}")
	#monthly_load_pct = go.Scatter(x=df_mf["Month"], y=df_mf["Mo Brk"], name=f"Monthly % Load, {ls_name[-4:]}, {version}", mode='lines+markers', line=dict(shape='spline'))
	pdaf = go.Scatter(x=df_mf["Month"], y=df_mf["PDAF"], name=f"Peak day/week day Ratio, {ls_name}, {version}", mode='lines+markers', line=dict(shape='spline'))
	wkend_wkday = go.Scatter(x=df_mf["Month"], y=df_mf["Wknd/Wkd"], name=f"Weekend/weekday Ratio, {ls_name}, {version}", mode='lines+markers', line=dict(shape='spline'))
	monthly_set = [monthly_load_pct, pdaf, wkend_wkday]
	return monthly_set




def prepare_plots(df, rdf, shape_selection, version):
	"""
	Inputs: df is the filtered df, rdf is filtered ratio df, shape selection is the widget dropdown for peak/avg/end,
	version is any identifier string used to distinguish between different sets of load shapes.
	Outputs: Plotly traces of seasonal hourly load shapes and monthly factors.
	
	:df: filtered cdls(hourly load shape) dataframe, produced using dataFilter() (pandas dataframe)
	:rdf: filtered ratios(monthly factors) dataframe, produced using dataFilter() (pandas dataframe)
	:shape_selection: dropdown object (created by produce_dropdowns()) for selecting peak/wkday/wkend shape (Jupyter dropdown widget)
	:version: any string value to distinguish this set of load shapes from other sets, for example: A, B or C (string)
	"""

	if len(df["BldgLoc"].unique())==1:
		ls_name, df_any_traces = breakDownSeasonalTraces_py(df, version)
		ls_month1 = [df_any_traces[0][shape_selection]]
		ls_month4 = [df_any_traces[1][shape_selection]]
		ls_month7 = [df_any_traces[2][shape_selection]]
		ls_month10 = [df_any_traces[3][shape_selection]]

		monthly_factors = [rdf]
		any_mf_traces = monthlyTraces_py(monthly_factors[0], ls_name, version)
		monthly_percent = [any_mf_traces[0]]
		pdaf_ratio = [any_mf_traces[1]]
		wkendwkday_ratio = [any_mf_traces[2]]

	else:

		df_cz1 = df[df["BldgLoc"]=="CZ01"]
		df_cz2 = df[df["BldgLoc"]=="CZ02"]
		df_cz3 = df[df["BldgLoc"]=="CZ03"]
		df_cz4 = df[df["BldgLoc"]=="CZ04"]
		df_cz5 = df[df["BldgLoc"]=="CZ05"]
		df_cz6 = df[df["BldgLoc"]=="CZ06"]
		df_cz7 = df[df["BldgLoc"]=="CZ07"]
		df_cz8 = df[df["BldgLoc"]=="CZ08"]
		df_cz9 = df[df["BldgLoc"]=="CZ09"]
		df_cz10 = df[df["BldgLoc"]=="CZ10"]
		df_cz11 = df[df["BldgLoc"]=="CZ11"]
		df_cz12 = df[df["BldgLoc"]=="CZ12"]
		df_cz13 = df[df["BldgLoc"]=="CZ13"]
		df_cz14 = df[df["BldgLoc"]=="CZ14"]
		df_cz15 = df[df["BldgLoc"]=="CZ15"]
		df_cz16 = df[df["BldgLoc"]=="CZ16"]
		
		#hourly load shape plotting traces by climate-zone
		ls_name_cz1, df_cz1_traces = breakDownSeasonalTraces_py(df_cz1, version)
		ls_name_cz2, df_cz2_traces = breakDownSeasonalTraces_py(df_cz2, version)
		ls_name_cz3, df_cz3_traces = breakDownSeasonalTraces_py(df_cz3, version)
		ls_name_cz4, df_cz4_traces = breakDownSeasonalTraces_py(df_cz4, version)
		ls_name_cz5, df_cz5_traces = breakDownSeasonalTraces_py(df_cz5, version)
		ls_name_cz6, df_cz6_traces = breakDownSeasonalTraces_py(df_cz6, version)
		ls_name_cz7, df_cz7_traces = breakDownSeasonalTraces_py(df_cz7, version)
		ls_name_cz8, df_cz8_traces = breakDownSeasonalTraces_py(df_cz8, version)
		ls_name_cz9, df_cz9_traces = breakDownSeasonalTraces_py(df_cz9, version)
		ls_name_cz10, df_cz10_traces = breakDownSeasonalTraces_py(df_cz10, version)
		ls_name_cz11, df_cz11_traces = breakDownSeasonalTraces_py(df_cz11, version)
		ls_name_cz12, df_cz12_traces = breakDownSeasonalTraces_py(df_cz12, version)
		ls_name_cz13, df_cz13_traces = breakDownSeasonalTraces_py(df_cz13, version)
		ls_name_cz14, df_cz14_traces = breakDownSeasonalTraces_py(df_cz14, version)
		ls_name_cz15, df_cz15_traces = breakDownSeasonalTraces_py(df_cz15, version)
		ls_name_cz16, df_cz16_traces = breakDownSeasonalTraces_py(df_cz16, version)
		
		ls_month1 = [df_cz1_traces[0][shape_selection],
				df_cz2_traces[0][shape_selection],
				df_cz3_traces[0][shape_selection],
				df_cz4_traces[0][shape_selection],
				df_cz5_traces[0][shape_selection],
				df_cz6_traces[0][shape_selection],
				df_cz7_traces[0][shape_selection],
				df_cz8_traces[0][shape_selection],
				df_cz9_traces[0][shape_selection],
				df_cz10_traces[0][shape_selection],
				df_cz11_traces[0][shape_selection],
				df_cz12_traces[0][shape_selection],
				df_cz13_traces[0][shape_selection],
				df_cz14_traces[0][shape_selection],
				df_cz15_traces[0][shape_selection],
				df_cz16_traces[0][shape_selection]]
		
		ls_month4 = [df_cz1_traces[1][shape_selection],
				df_cz2_traces[1][shape_selection],
				df_cz3_traces[1][shape_selection],
				df_cz4_traces[1][shape_selection],
				df_cz5_traces[1][shape_selection],
				df_cz6_traces[1][shape_selection],
				df_cz7_traces[1][shape_selection],
				df_cz8_traces[1][shape_selection],
				df_cz9_traces[1][shape_selection],
				df_cz10_traces[1][shape_selection],
				df_cz11_traces[1][shape_selection],
				df_cz12_traces[1][shape_selection],
				df_cz13_traces[1][shape_selection],
				df_cz14_traces[1][shape_selection],
				df_cz15_traces[1][shape_selection],
				df_cz16_traces[1][shape_selection]]
		
		ls_month7 = [df_cz1_traces[2][shape_selection],
				df_cz2_traces[2][shape_selection],
				df_cz3_traces[2][shape_selection],
				df_cz4_traces[2][shape_selection],
				df_cz5_traces[2][shape_selection],
				df_cz6_traces[2][shape_selection],
				df_cz7_traces[2][shape_selection],
				df_cz8_traces[2][shape_selection],
				df_cz9_traces[2][shape_selection],
				df_cz10_traces[2][shape_selection],
				df_cz11_traces[2][shape_selection],
				df_cz12_traces[2][shape_selection],
				df_cz13_traces[2][shape_selection],
				df_cz14_traces[2][shape_selection],
				df_cz15_traces[2][shape_selection],
				df_cz16_traces[2][shape_selection]]
		
		ls_month10 = [df_cz1_traces[3][shape_selection],
				df_cz2_traces[3][shape_selection],
				df_cz3_traces[3][shape_selection],
				df_cz4_traces[3][shape_selection],
				df_cz5_traces[3][shape_selection],
				df_cz6_traces[3][shape_selection],
				df_cz7_traces[3][shape_selection],
				df_cz8_traces[3][shape_selection],
				df_cz9_traces[3][shape_selection],
				df_cz10_traces[3][shape_selection],
				df_cz11_traces[3][shape_selection],
				df_cz12_traces[3][shape_selection],
				df_cz13_traces[3][shape_selection],
				df_cz14_traces[3][shape_selection],
				df_cz15_traces[3][shape_selection],
				df_cz16_traces[3][shape_selection]]
		
		
		#monthly factors by climate zone
		monthly_factors = assembleMonthlyFactor_py(rdf)
		cz1_mf_traces = monthlyTraces_py(monthly_factors[0], ls_name_cz1, version)
		cz2_mf_traces = monthlyTraces_py(monthly_factors[1], ls_name_cz2, version)
		cz3_mf_traces = monthlyTraces_py(monthly_factors[2], ls_name_cz3, version)
		cz4_mf_traces = monthlyTraces_py(monthly_factors[3], ls_name_cz4, version)
		cz5_mf_traces = monthlyTraces_py(monthly_factors[4], ls_name_cz5, version)
		cz6_mf_traces = monthlyTraces_py(monthly_factors[5], ls_name_cz6, version)
		cz7_mf_traces = monthlyTraces_py(monthly_factors[6], ls_name_cz7, version)
		cz8_mf_traces = monthlyTraces_py(monthly_factors[7], ls_name_cz8, version)
		cz9_mf_traces = monthlyTraces_py(monthly_factors[8], ls_name_cz9, version)
		cz10_mf_traces = monthlyTraces_py(monthly_factors[9], ls_name_cz10, version)
		cz11_mf_traces = monthlyTraces_py(monthly_factors[10], ls_name_cz11, version)
		cz12_mf_traces = monthlyTraces_py(monthly_factors[11], ls_name_cz12, version)
		cz13_mf_traces = monthlyTraces_py(monthly_factors[12], ls_name_cz13, version)
		cz14_mf_traces = monthlyTraces_py(monthly_factors[13], ls_name_cz14, version)
		cz15_mf_traces = monthlyTraces_py(monthly_factors[14], ls_name_cz15, version)
		cz16_mf_traces = monthlyTraces_py(monthly_factors[15], ls_name_cz16, version)
		
		monthly_percent = [cz1_mf_traces[0],
						   cz2_mf_traces[0],
						   cz3_mf_traces[0],
						   cz4_mf_traces[0],
						   cz5_mf_traces[0],
						   cz6_mf_traces[0],
						   cz7_mf_traces[0],
						   cz8_mf_traces[0],
						   cz9_mf_traces[0],
						   cz10_mf_traces[0],
						   cz11_mf_traces[0],
						   cz12_mf_traces[0],
						   cz13_mf_traces[0],
						   cz14_mf_traces[0],
						   cz15_mf_traces[0],
						   cz16_mf_traces[0]]
		
		pdaf_ratio = [cz1_mf_traces[1],
						   cz2_mf_traces[1],
						   cz3_mf_traces[1],
						   cz4_mf_traces[1],
						   cz5_mf_traces[1],
						   cz6_mf_traces[1],
						   cz7_mf_traces[1],
						   cz8_mf_traces[1],
						   cz9_mf_traces[1],
						   cz10_mf_traces[1],
						   cz11_mf_traces[1],
						   cz12_mf_traces[1],
						   cz13_mf_traces[1],
						   cz14_mf_traces[1],
						   cz15_mf_traces[1],
						   cz16_mf_traces[1]]
		
		wkendwkday_ratio = [cz1_mf_traces[2],
						   cz2_mf_traces[2],
						   cz3_mf_traces[2],
						   cz4_mf_traces[2],
						   cz5_mf_traces[2],
						   cz6_mf_traces[2],
						   cz7_mf_traces[2],
						   cz8_mf_traces[2],
						   cz9_mf_traces[2],
						   cz10_mf_traces[2],
						   cz11_mf_traces[2],
						   cz12_mf_traces[2],
						   cz13_mf_traces[2],
						   cz14_mf_traces[2],
						   cz15_mf_traces[2],
						   cz16_mf_traces[2]]
		
	hourly_ls = [ls_month1, ls_month4, ls_month7, ls_month10]
	monthly_ratios = [monthly_percent, pdaf_ratio, wkendwkday_ratio]
	
	return hourly_ls, monthly_ratios




def plot_seasons(df, version):
	"""
	input the collection of 4 seasonal ls produced by prepare_plots(), and plot them using plotly, on a jupyter notebook 

	:df: collection of hourly ls plotting traces produced by prepare_plots() (a list created by plotly traces)
	:version: any string value to distinguish this set of load shapes from other sets, for example: A, B or C (string)

	"""
	m1 = df[0]
	m4 = df[1]
	m7 = df[2]
	m10 = df[3]
	fig = make_subplots(rows=2, cols=2, subplot_titles=("January","July","April","October"))
	fig.update_layout(title_text=f"seasonal loadshapes, group {version}")
	
	#plot the 4 seasons
	for i in range(0, len(m1)):
		fig.add_trace(m1[i], row=1, col=1)
	for i in range(0, len(m4)):
		fig.add_trace(m4[i], row=2, col=1)
	for i in range(0, len(m7)):
		fig.add_trace(m7[i], row=1, col=2)
	for i in range(0, len(m10)):
		fig.add_trace(m10[i], row=2, col=2)

	return fig
	#fig.show()



def plot_detail(df_a, df_b, techid_a, techid_b, bldgtype_a, bldgtype_b, shape_a, shape_b, deerver_a, deerver_b):
	"""
	input two seasonal ls sets produced by prepare_plots, and plot them all in the same plot using plotly, also produce dropdowns to filter the plots

	:df_a: collection of hourly ls plotting traces produced by prepare_plots(), set A (a list created by plotly traces)
	:df_b: collection of hourly ls plotting traces produced by prepare_plots(), set B (a list created by plotly traces)
	:techid_a: tech ID of set A (string)
	:techid_b: tech ID of set B (string)
	:bldgtype_a: building type of set A (string)
	:bldgtype_b: building type of set B (string)
	:shape_a: shape type (peak, wkday, wkend) of set A (string)
	:shape_b: shape type (peak, wkday, wkend) of set B (string)
	:deerver_a: deer version of set A (string)
	:deerver_b: deer version of set B (string)
	"""
	month1 = df_a[0] + df_b[0]
	month4 = df_a[1] + df_b[1]
	month7 = df_a[2] + df_b[2]
	month10 = df_a[3] + df_b[3]
	month_df_list = (month1, month4, month7, month10)
	#hourly plots logic mapping

	if ((len(df_a[0])== 1) or (len(df_b[0]) == 1)) and ((len(df_a[0])== 16) or (len(df_b[0]) == 16)):
		x = list(range(0,68)) #only 17 shapes per month, x 4 months, if one set is "Any"
		all_hide_bool = x.copy()       
		for i in range(0,68):
			all_hide_bool[i] = False

		month1_bool = all_hide_bool.copy()
		month4_bool = all_hide_bool.copy()
		month7_bool = all_hide_bool.copy()
		month10_bool = all_hide_bool.copy()

		for i in range(0,17):
			month1_bool[i]= True
		for i in range(17,34):
			month4_bool[i]= True
		for i in range(34,51):
			month7_bool[i]= True
		for i in range(51,68):
			month10_bool[i]= True

	elif ((len(df_a[0])== 1) and (len(df_b[0]) == 1)):
		x = list(range(0,8)) #only 2 shapes per month, x 4 months, if both set is "Any"
		all_hide_bool = x.copy()
		for i in range(0,8):
			all_hide_bool[i] = False

		month1_bool = all_hide_bool.copy()
		month4_bool = all_hide_bool.copy()
		month7_bool = all_hide_bool.copy()
		month10_bool = all_hide_bool.copy()

		for i in range(0,2):
			month1_bool[i]= True
		for i in range(2,4):
			month4_bool[i]= True
		for i in range(4,6):
			month7_bool[i]= True
		for i in range(6,8):
			month10_bool[i]= True

	else:

		x = list(range(0,128)) #16+16 = 32 shapes per month x 4 months

		all_hide_bool = x.copy()       
		for i in range(0,128):
			all_hide_bool[i] = False

		month1_bool = all_hide_bool.copy()
		month4_bool = all_hide_bool.copy()
		month7_bool = all_hide_bool.copy()
		month10_bool = all_hide_bool.copy()

		for i in range(0,32):
			month1_bool[i]= True
		for i in range(32,64):
			month4_bool[i]= True
		for i in range(64,96):
			month7_bool[i]= True
		for i in range(96,128):
			month10_bool[i]= True
		
	layout = go.Layout(title=f"Hourly Load Shapes All<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | {shape_a} | Version: {deerver_a}" + 
						    f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | {shape_b} | Version: {deerver_b}" + "<br> ", 
				   yaxis=dict(title='load %'),
				   xaxis=dict(title='hour of day'),
				   hoverlabel = dict(namelength=-1))
	
	fig = go.Figure(layout = layout)
	#plot every trace (32*4=128) on one plot, then apply filter booleans using "update" method
	for month in range(0,len(month_df_list)):
		for i in range(0,len(month_df_list[month])):
			fig.add_trace(month_df_list[month][i])
	fig.update_layout(title={'x':0.5})	
	
	fig.update_layout(
	annotations=[dict(text='select month:',
					  showarrow=False,
					  x=-0.1, y=1, xref="paper", yref="paper", align="left", xanchor="left", #yanchor="top"
					 )
				],
	updatemenus=[
		dict(
			active=0,
			buttons=list([
				dict(label="All",
					 method="update",
					 args=[{"visible": [True]},
						   {"title": f"Hourly Load Shapes All<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | {shape_a} | Version: {deerver_a}" + 
						    f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | {shape_b} | Version: {deerver_b}" + "<br> "}]),
				dict(label="January",
					 method="update",
					 args=[{"visible": month1_bool},
						   {"title": f"Hourly Load Shapes January<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | {shape_a} | Version: {deerver_a}" + 
						   f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | {shape_b} | Version: {deerver_b}" + "<br> "}]),
				dict(label="April",
					 method="update",
					 args=[{"visible": month4_bool},
						   {"title": f"Hourly Load Shapes April<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | {shape_a} | Version: {deerver_a}" + 
						   f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | {shape_b} | Version: {deerver_b}" + "<br> "}]),
				dict(label="July",
					 method="update",
					 args=[{"visible": month7_bool},
						   {"title": f"Hourly Load Shapes July<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | {shape_a} | Version: {deerver_a}" + 
						   f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | {shape_b} | Version: {deerver_b}" + "<br> "}]),
				dict(label="October",
					 method="update",
					 args=[{"visible": month10_bool},
						   {"title": f"Hourly Load Shapes October<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | {shape_a} | Version: {deerver_a}" + 
						   f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | {shape_b} | Version: {deerver_b}" + "<br> "}]),
				
			]),
			x=-0.1,
			xanchor="left",
			y=0.95,
			yanchor="top"
			)
	])
	#fig.show()
	return fig



def plot_detail_single(df_a, techid, bldgtype):
	"""
	input one seasonal ls sets produced by prepare_plots, and plot them all in the same plot using plotly, also produce dropdowns to filter the plots

	:df_a: collection of hourly ls plotting traces produced by prepare_plots(), set A (a list created by plotly traces)
	:techid: string for measure Tech ID
	:bldgtype: string for bldgtype
	"""
	month1 = df_a[0]
	month4 = df_a[1]
	month7 = df_a[2]
	month10 = df_a[3]
	month_df_list = (month1, month4, month7, month10)

	if (len(df_a[0]) == 1):
		layout = go.Layout(title=f'detailed loadshapes for Tech ID: {techid}, BldgType: {bldgtype} (All CZ)', 
				   yaxis=dict(title='load %'),
				   xaxis=dict(title='hour of day'),
				   hoverlabel = dict(namelength=-1))
	
		fig = go.Figure(layout = layout)
		fig.update_layout(title={'xanchor':'center', 'yanchor':'top'})
		#plot every trace on one plot
		for month in range(0,len(month_df_list)):
			for i in range(0,len(month_df_list[month])):
				fig.add_trace(month_df_list[month][i])
		#fig.show()
		return fig

	else:
		#hourly plots logic mapping
		x = list(range(0,64))

		all_hide_bool = x.copy()       
		for i in range(0,64):
			all_hide_bool[i] = False

		month1_bool = all_hide_bool.copy()
		month4_bool = all_hide_bool.copy()
		month7_bool = all_hide_bool.copy()
		month10_bool = all_hide_bool.copy()

		for i in range(0,16):
			month1_bool[i]= True
		for i in range(16,32):
			month4_bool[i]= True
		for i in range(32,48):
			month7_bool[i]= True
		for i in range(48,64):
			month10_bool[i]= True
			
		layout = go.Layout(title=f'detailed loadshapes for Tech ID: {techid}, BldgType: {bldgtype} (All CZ)', 
					   yaxis=dict(title='load %'),
					   xaxis=dict(title='hour of day'),
					   hoverlabel = dict(namelength=-1))
		
		fig = go.Figure(layout = layout)
		#plot every trace (32*4=128) on one plot, then apply filter booleans using "update" method
		for month in range(0,len(month_df_list)):
			for i in range(0,len(month_df_list[month])):
				fig.add_trace(month_df_list[month][i])
			
		
		fig.update_layout(

		updatemenus=[
			dict(
				active=0,
				buttons=list([
					dict(label="All",
						 method="update",
						 args=[{"visible": [True]},
							   {"title": f"detailed loadshapes for Tech ID: {techid}, BldgType: {bldgtype} (All CZ)"}]),
					dict(label="January",
						 method="update",
						 args=[{"visible": month1_bool},
							   {"title": f"January loadshapes for Tech ID: {techid}, BldgType: {bldgtype} (All CZ)"}]),
					dict(label="April",
						 method="update",
						 args=[{"visible": month4_bool},
							   {"title": f"April loadshapes for Tech ID: {techid}, BldgType: {bldgtype} (All CZ)"}]),
					dict(label="July",
						 method="update",
						 args=[{"visible": month7_bool},
							   {"title": f"July loadshapes for Tech ID: {techid}, BldgType: {bldgtype} (All CZ)"}]),
					dict(label="October",
						 method="update",
						 args=[{"visible": month10_bool},
							   {"title": f"October loadshapes for Tech ID: {techid}, BldgType: {bldgtype} (All CZ)"}]),
					
				]),
			)
		])
		#fig.show()
		return fig


def plot_monthly_factors(df_a, df_b, techid_a, techid_b, bldgtype_a, bldgtype_b, deerver_a, deerver_b):
	"""
	input monthly_ratio sets produced by prepare_plots(), and plot them using plotly
	
	:df_a: collection of monthly ratios plotting traces produced by prepare_plots(), set A (a list created by plotly traces)
	:df_b: collection of monthly ratios plotting traces produced by prepare_plots(), set B (a list created by plotly traces)
	"""
	monthly_percent = df_a[0] + df_b[0]
	pdaf_ratio = df_a[1] + df_b[1]
	wkendwkday_ratio = df_a[2] + df_b[2]
	
	#monthly plots logic mapping
	if ((len(df_a[0])== 1) or (len(df_b[0]) == 1)) and ((len(df_a[0])== 16) or (len(df_b[0]) == 16)):

		x = list(range(0,17)) #only 17 sets of ratios (16+1) per plot if one set is "Any"
		
		monthly_all_hide = x.copy()       
		for i in range(0,17):
			monthly_all_hide[i] = False

		plot_cz1 = monthly_all_hide.copy()
		plot_cz2 = monthly_all_hide.copy()
		plot_cz3 = monthly_all_hide.copy()
		plot_cz4 = monthly_all_hide.copy()
		plot_cz5 = monthly_all_hide.copy()
		plot_cz6 = monthly_all_hide.copy()
		plot_cz7 = monthly_all_hide.copy()
		plot_cz8 = monthly_all_hide.copy()
		plot_cz9 = monthly_all_hide.copy()
		plot_cz10 = monthly_all_hide.copy()
		plot_cz11 = monthly_all_hide.copy()
		plot_cz12 = monthly_all_hide.copy()
		plot_cz13 = monthly_all_hide.copy()
		plot_cz14 = monthly_all_hide.copy()
		plot_cz15 = monthly_all_hide.copy()
		plot_cz16 = monthly_all_hide.copy()
		plot_A = monthly_all_hide.copy()
		plot_B = monthly_all_hide.copy()

		#boolean list to determine which to show
		if len(df_a[0]) == 1:
			plot_cz1[0]=True
			plot_cz1[1]=True
			plot_cz2[0]=True
			plot_cz2[2]=True
			plot_cz3[0]=True
			plot_cz3[3]=True
			plot_cz4[0]=True
			plot_cz4[4]=True
			plot_cz5[0]=True
			plot_cz5[5]=True
			plot_cz6[0]=True
			plot_cz6[6]=True
			plot_cz7[0]=True
			plot_cz7[7]=True
			plot_cz8[0]=True
			plot_cz8[8]=True
			plot_cz9[0]=True
			plot_cz9[9]=True
			plot_cz10[0]=True
			plot_cz10[10]=True
			plot_cz11[0]=True
			plot_cz11[11]=True
			plot_cz12[0]=True
			plot_cz12[12]=True
			plot_cz13[0]=True
			plot_cz13[13]=True
			plot_cz14[0]=True
			plot_cz14[14]=True
			plot_cz15[0]=True
			plot_cz15[15]=True
			plot_cz16[0]=True
			plot_cz16[16]=True

			for i in range(0,1):
				plot_A[i]= True
			for i in range(1,17):
				plot_B[i]= True

		elif len(df_b[0]) == 1:
			plot_cz1[0]=True
			plot_cz1[16]=True
			plot_cz2[1]=True
			plot_cz2[16]=True
			plot_cz3[2]=True
			plot_cz3[16]=True
			plot_cz4[3]=True
			plot_cz4[16]=True
			plot_cz5[4]=True
			plot_cz5[16]=True
			plot_cz6[5]=True
			plot_cz6[16]=True
			plot_cz7[6]=True
			plot_cz7[16]=True
			plot_cz8[7]=True
			plot_cz8[16]=True
			plot_cz9[8]=True
			plot_cz9[16]=True
			plot_cz10[9]=True
			plot_cz10[16]=True
			plot_cz11[10]=True
			plot_cz11[16]=True
			plot_cz12[11]=True
			plot_cz12[16]=True
			plot_cz13[12]=True
			plot_cz13[16]=True
			plot_cz14[13]=True
			plot_cz14[16]=True
			plot_cz15[14]=True
			plot_cz15[16]=True
			plot_cz16[15]=True
			plot_cz16[16]=True

			for i in range(0,16):
				plot_A[i]= True
			for i in range(16,17):
				plot_B[i]= True

	elif ((len(df_a[0])== 1) and (len(df_b[0]) == 1)):

		layout_percentload = go.Layout(title=f"All Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
											f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> ", 
							   yaxis=dict(title='monthly load %'),
							   xaxis=dict(title='months'),
							   hoverlabel = dict(namelength=-1))

		layout_pdaf = go.Layout(title=f"All Peak day/weekday ratio <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
									f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> ",
								yaxis=dict(title='ratio'),
								xaxis=dict(title='months'),
								hoverlabel = dict(namelength=-1))

		layout_wkendwkday = go.Layout(title=f"All weekend/weekday ratio<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
											f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> ",
									  yaxis=dict(title='ratio'),
									  xaxis=dict(title='months'),
									  hoverlabel = dict(namelength=-1))

		fig4 = go.Figure(data=monthly_percent,layout=layout_percentload)
		fig5 = go.Figure(data=pdaf_ratio,layout=layout_pdaf)
		fig6 = go.Figure(data=wkendwkday_ratio,layout=layout_wkendwkday)
		# fig4.show()
		# fig5.show()
		# fig6.show()
		return fig4, fig5, fig6

	else:

		x = list(range(0,32))

		monthly_all_hide = x.copy()       
		for i in range(0,32):
			monthly_all_hide[i] = False

		plot_cz1 = monthly_all_hide.copy()
		plot_cz2 = monthly_all_hide.copy()
		plot_cz3 = monthly_all_hide.copy()
		plot_cz4 = monthly_all_hide.copy()
		plot_cz5 = monthly_all_hide.copy()
		plot_cz6 = monthly_all_hide.copy()
		plot_cz7 = monthly_all_hide.copy()
		plot_cz8 = monthly_all_hide.copy()
		plot_cz9 = monthly_all_hide.copy()
		plot_cz10 = monthly_all_hide.copy()
		plot_cz11 = monthly_all_hide.copy()
		plot_cz12 = monthly_all_hide.copy()
		plot_cz13 = monthly_all_hide.copy()
		plot_cz14 = monthly_all_hide.copy()
		plot_cz15 = monthly_all_hide.copy()
		plot_cz16 = monthly_all_hide.copy()
		plot_A = monthly_all_hide.copy()
		plot_B = monthly_all_hide.copy()

		#boolean list to determine which to show
		plot_cz1[0]=True
		plot_cz1[16]=True
		plot_cz2[1]=True
		plot_cz2[17]=True
		plot_cz3[2]=True
		plot_cz3[18]=True
		plot_cz4[3]=True
		plot_cz4[19]=True
		plot_cz5[4]=True
		plot_cz5[20]=True
		plot_cz6[5]=True
		plot_cz6[21]=True
		plot_cz7[6]=True
		plot_cz7[22]=True
		plot_cz8[7]=True
		plot_cz8[23]=True
		plot_cz9[8]=True
		plot_cz9[24]=True
		plot_cz10[9]=True
		plot_cz10[25]=True
		plot_cz11[10]=True
		plot_cz11[26]=True
		plot_cz12[11]=True
		plot_cz12[27]=True
		plot_cz13[12]=True
		plot_cz13[28]=True
		plot_cz14[13]=True
		plot_cz14[29]=True
		plot_cz15[14]=True
		plot_cz15[30]=True
		plot_cz16[15]=True
		plot_cz16[31]=True

		for i in range(0,16):
			plot_A[i]= True
		for i in range(16,32):
			plot_B[i]= True
	


	if ((len(df_a[0])== 1) and (len(df_b[0]) == 1)):
		pass

	else:
		layout_percentload = go.Layout(title=f"All Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
												 f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> ", 
								   yaxis=dict(title='monthly load %'),
								   xaxis=dict(title='months'),
								   hoverlabel = dict(namelength=-1))

		layout_pdaf = go.Layout(title=f"All Peak day/weekday ratio <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
									 f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> ",
								yaxis=dict(title='ratio'),
								xaxis=dict(title='months'),
								hoverlabel = dict(namelength=-1))

		layout_wkendwkday = go.Layout(title=f"All weekend/weekday ratio<br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
											f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> ",
									  yaxis=dict(title='ratio'),
									  xaxis=dict(title='months'),
									  hoverlabel = dict(namelength=-1))

		fig4 = go.Figure(data=monthly_percent,layout=layout_percentload)
		fig5 = go.Figure(data=pdaf_ratio,layout=layout_pdaf)
		fig6 = go.Figure(data=wkendwkday_ratio,layout=layout_wkendwkday)

		fig4.update_layout(
			annotations=[dict(text='select climate<br>zone:',
					  showarrow=False,
					  x=-0.1, y=1, xref="paper", yref="paper", align="left", xanchor="left", #yanchor="top"
					 		  )
						],
			updatemenus=[
				dict(
					active=0,
					buttons=list([ #3/2/21: update these plot titles
						dict(label="All", method="update", args=[{"visible": [True]}, {"title": f"All Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
												 f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="All set A", method="update", args=[{"visible": plot_A}, {"title": f"All Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" }]),
						dict(label="All set B", method="update", args=[{"visible": plot_B}, {"title": f"All Monthly Load % <br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}"}]),
						dict(label="CZ01", method="update", args=[{"visible": plot_cz1}, {"title": f"CZ01 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ02", method="update", args=[{"visible": plot_cz2}, {"title": f"CZ02 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ03", method="update", args=[{"visible": plot_cz3}, {"title": f"CZ03 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ04", method="update", args=[{"visible": plot_cz4}, {"title": f"CZ04 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ05", method="update", args=[{"visible": plot_cz5}, {"title": f"CZ05 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ06", method="update", args=[{"visible": plot_cz6}, {"title": f"CZ06 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ07", method="update", args=[{"visible": plot_cz7}, {"title": f"CZ07 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ08", method="update", args=[{"visible": plot_cz8}, {"title": f"CZ08 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ09", method="update", args=[{"visible": plot_cz9}, {"title": f"CZ09 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ10", method="update", args=[{"visible": plot_cz10}, {"title": f"CZ10 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ11", method="update", args=[{"visible": plot_cz11}, {"title": f"CZ11 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ12", method="update", args=[{"visible": plot_cz12}, {"title": f"CZ12 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ13", method="update", args=[{"visible": plot_cz13}, {"title": f"CZ13 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ14", method="update", args=[{"visible": plot_cz14}, {"title": f"CZ14 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ15", method="update", args=[{"visible": plot_cz15}, {"title": f"CZ15 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),
						dict(label="CZ16", method="update", args=[{"visible": plot_cz16}, {"title": f"CZ16 Monthly Load % <br>Set A - Tech ID: {techid_a} | Bldg Type: {bldgtype_a} | Version: {deerver_a}" + 
																								  	f"<br>Set B - Tech ID: {techid_b} | Bldg Type: {bldgtype_b} | Version: {deerver_b}" + "<br> "}]),   
								]),
					x=-0.1,
					xanchor="left",
					y=0.95,
					yanchor="top"
					 )
						 ])

		fig5.update_layout(
			annotations=[dict(text='select climate zone:',
					  showarrow=False,
					  x=-0.1, y=1, xref="paper", yref="paper", align="left", xanchor="left", #yanchor="top"
					 		  )
						],
			updatemenus=[
				dict(
					active=0,
					buttons=list([
						dict(label="All", method="update", args=[{"visible": [True]}, {"title": "All Peak day/weekday ratio"}]),
						dict(label="All set A", method="update", args=[{"visible": plot_A}, {"title": "All CZs, set A"}]),
						dict(label="All set B", method="update", args=[{"visible": plot_B}, {"title": "All CZs, set B"}]),
						dict(label="CZ01", method="update", args=[{"visible": plot_cz1}, {"title": "CZ01 comparison"}]),
						dict(label="CZ02", method="update", args=[{"visible": plot_cz2}, {"title": "CZ02 comparison"}]),
						dict(label="CZ03", method="update", args=[{"visible": plot_cz3}, {"title": "CZ03 comparison"}]),
						dict(label="CZ04", method="update", args=[{"visible": plot_cz4}, {"title": "CZ04 comparison"}]),
						dict(label="CZ05", method="update", args=[{"visible": plot_cz5}, {"title": "CZ05 comparison"}]),
						dict(label="CZ06", method="update", args=[{"visible": plot_cz6}, {"title": "CZ06 comparison"}]),
						dict(label="CZ07", method="update", args=[{"visible": plot_cz7}, {"title": "CZ07 comparison"}]),
						dict(label="CZ08", method="update", args=[{"visible": plot_cz8}, {"title": "CZ08 comparison"}]),
						dict(label="CZ09", method="update", args=[{"visible": plot_cz9}, {"title": "CZ09 comparison"}]),
						dict(label="CZ10", method="update", args=[{"visible": plot_cz10}, {"title": "CZ10 comparison"}]),
						dict(label="CZ11", method="update", args=[{"visible": plot_cz11}, {"title": "CZ11 comparison"}]),
						dict(label="CZ12", method="update", args=[{"visible": plot_cz12}, {"title": "CZ12 comparison"}]),
						dict(label="CZ13", method="update", args=[{"visible": plot_cz13}, {"title": "CZ13 comparison"}]),
						dict(label="CZ14", method="update", args=[{"visible": plot_cz14}, {"title": "CZ14 comparison"}]),
						dict(label="CZ15", method="update", args=[{"visible": plot_cz15}, {"title": "CZ15 comparison"}]),
						dict(label="CZ16", method="update", args=[{"visible": plot_cz16}, {"title": "CZ16 comparison"}]),   
								]),
					x=-0.1,
					xanchor="left",
					y=0.95,
					yanchor="top"
					 )
						 ])

		fig6.update_layout(
			annotations=[dict(text='select climate zone:',
					  showarrow=False,
					  x=-0.1, y=1, xref="paper", yref="paper", align="left", xanchor="left", #yanchor="top"
					 		  )
						],
			updatemenus=[
				dict(
					active=0,
					buttons=list([
						dict(label="All", method="update", args=[{"visible": [True]}, {"title": "All weekend/weekday ratio"}]),
						dict(label="All set A", method="update", args=[{"visible": plot_A}, {"title": "All CZs, set A"}]),
						dict(label="All set B", method="update", args=[{"visible": plot_B}, {"title": "All CZs, set B"}]),
						dict(label="CZ01", method="update", args=[{"visible": plot_cz1}, {"title": "CZ01 comparison"}]),
						dict(label="CZ02", method="update", args=[{"visible": plot_cz2}, {"title": "CZ02 comparison"}]),
						dict(label="CZ03", method="update", args=[{"visible": plot_cz3}, {"title": "CZ03 comparison"}]),
						dict(label="CZ04", method="update", args=[{"visible": plot_cz4}, {"title": "CZ04 comparison"}]),
						dict(label="CZ05", method="update", args=[{"visible": plot_cz5}, {"title": "CZ05 comparison"}]),
						dict(label="CZ06", method="update", args=[{"visible": plot_cz6}, {"title": "CZ06 comparison"}]),
						dict(label="CZ07", method="update", args=[{"visible": plot_cz7}, {"title": "CZ07 comparison"}]),
						dict(label="CZ08", method="update", args=[{"visible": plot_cz8}, {"title": "CZ08 comparison"}]),
						dict(label="CZ09", method="update", args=[{"visible": plot_cz9}, {"title": "CZ09 comparison"}]),
						dict(label="CZ10", method="update", args=[{"visible": plot_cz10}, {"title": "CZ10 comparison"}]),
						dict(label="CZ11", method="update", args=[{"visible": plot_cz11}, {"title": "CZ11 comparison"}]),
						dict(label="CZ12", method="update", args=[{"visible": plot_cz12}, {"title": "CZ12 comparison"}]),
						dict(label="CZ13", method="update", args=[{"visible": plot_cz13}, {"title": "CZ13 comparison"}]),
						dict(label="CZ14", method="update", args=[{"visible": plot_cz14}, {"title": "CZ14 comparison"}]),
						dict(label="CZ15", method="update", args=[{"visible": plot_cz15}, {"title": "CZ15 comparison"}]),
						dict(label="CZ16", method="update", args=[{"visible": plot_cz16}, {"title": "CZ16 comparison"}]),   
								]),
					x=-0.1,
					xanchor="left",
					y=0.95,
					yanchor="top"
					 )
						 ])

		#fig4.show()
		#fig5.show()
		#fig6.show()
		return fig4, fig5, fig6




def plot_monthly_factors_single(df_a):
	"""
	input monthly_ratio sets produced by prepare_plots(), and plot them using plotly
	
	:df_a: collection of monthly ratios plotting traces produced by prepare_plots(), set A (a list created by plotly traces)
	"""
	monthly_percent = df_a[0]
	pdaf_ratio = df_a[1]
	wkendwkday_ratio = df_a[2]
	
	#monthly plots logic mapping
	x = list(range(0,16))

	monthly_all_hide = x.copy()       
	for i in range(0,16):
		monthly_all_hide[i] = False

	plot_cz1 = monthly_all_hide.copy()
	plot_cz2 = monthly_all_hide.copy()
	plot_cz3 = monthly_all_hide.copy()
	plot_cz4 = monthly_all_hide.copy()
	plot_cz5 = monthly_all_hide.copy()
	plot_cz6 = monthly_all_hide.copy()
	plot_cz7 = monthly_all_hide.copy()
	plot_cz8 = monthly_all_hide.copy()
	plot_cz9 = monthly_all_hide.copy()
	plot_cz10 = monthly_all_hide.copy()
	plot_cz11 = monthly_all_hide.copy()
	plot_cz12 = monthly_all_hide.copy()
	plot_cz13 = monthly_all_hide.copy()
	plot_cz14 = monthly_all_hide.copy()
	plot_cz15 = monthly_all_hide.copy()
	plot_cz16 = monthly_all_hide.copy()
	plot_A = monthly_all_hide.copy()
	#plot_B = monthly_all_hide.copy()

	#boolean list to determine which to show
	plot_cz1[0]=True
	# plot_cz1[16]=True
	plot_cz2[1]=True
	# plot_cz2[17]=True
	plot_cz3[2]=True
	# plot_cz3[18]=True
	plot_cz4[3]=True
	# plot_cz4[19]=True
	plot_cz5[4]=True
	# plot_cz5[20]=True
	plot_cz6[5]=True
	# plot_cz6[21]=True
	plot_cz7[6]=True
	# plot_cz7[22]=True
	plot_cz8[7]=True
	# plot_cz8[23]=True
	plot_cz9[8]=True
	# plot_cz9[24]=True
	plot_cz10[9]=True
	# plot_cz10[25]=True
	plot_cz11[10]=True
	# plot_cz11[26]=True
	plot_cz12[11]=True
	# plot_cz12[27]=True
	plot_cz13[12]=True
	# plot_cz13[28]=True
	plot_cz14[13]=True
	# plot_cz14[29]=True
	plot_cz15[14]=True
	# plot_cz15[30]=True
	plot_cz16[15]=True
	# plot_cz16[31]=True

	for i in range(0,16):
		plot_A[i]= True
	# for i in range(16,32):
	# 	plot_B[i]= True
	
	layout_percentload = go.Layout(title='All Monthly Load %', 
							   yaxis=dict(title='monthly load %'),
							   xaxis=dict(title='months'),
							   hoverlabel = dict(namelength=-1))

	layout_pdaf = go.Layout(title='All Peak day/weekday ratio',
							yaxis=dict(title='ratio'),
							xaxis=dict(title='months'),
							hoverlabel = dict(namelength=-1))

	layout_wkendwkday = go.Layout(title='All weekend/weekday ratio',
								  yaxis=dict(title='ratio'),
								  xaxis=dict(title='months'),
								  hoverlabel = dict(namelength=-1))

	fig4 = go.Figure(data=monthly_percent,layout=layout_percentload)
	fig5 = go.Figure(data=pdaf_ratio,layout=layout_pdaf)
	fig6 = go.Figure(data=wkendwkday_ratio,layout=layout_wkendwkday)

	fig4.update_layout(
		updatemenus=[
			dict(
				active=0,
				buttons=list([
					dict(label="All", method="update", args=[{"visible": [True]}, {"title": "All Monthly Load %"}]),
					dict(label="All set A", method="update", args=[{"visible": plot_A}, {"title": "All CZs, set A"}]),
					#dict(label="All set B", method="update", args=[{"visible": plot_B}, {"title": "All CZs, set B"}]),
					dict(label="CZ01", method="update", args=[{"visible": plot_cz1}, {"title": "CZ01 comparison"}]),
					dict(label="CZ02", method="update", args=[{"visible": plot_cz2}, {"title": "CZ02 comparison"}]),
					dict(label="CZ03", method="update", args=[{"visible": plot_cz3}, {"title": "CZ03 comparison"}]),
					dict(label="CZ04", method="update", args=[{"visible": plot_cz4}, {"title": "CZ04 comparison"}]),
					dict(label="CZ05", method="update", args=[{"visible": plot_cz5}, {"title": "CZ05 comparison"}]),
					dict(label="CZ06", method="update", args=[{"visible": plot_cz6}, {"title": "CZ06 comparison"}]),
					dict(label="CZ07", method="update", args=[{"visible": plot_cz7}, {"title": "CZ07 comparison"}]),
					dict(label="CZ08", method="update", args=[{"visible": plot_cz8}, {"title": "CZ08 comparison"}]),
					dict(label="CZ09", method="update", args=[{"visible": plot_cz9}, {"title": "CZ09 comparison"}]),
					dict(label="CZ10", method="update", args=[{"visible": plot_cz10}, {"title": "CZ10 comparison"}]),
					dict(label="CZ11", method="update", args=[{"visible": plot_cz11}, {"title": "CZ11 comparison"}]),
					dict(label="CZ12", method="update", args=[{"visible": plot_cz12}, {"title": "CZ12 comparison"}]),
					dict(label="CZ13", method="update", args=[{"visible": plot_cz13}, {"title": "CZ13 comparison"}]),
					dict(label="CZ14", method="update", args=[{"visible": plot_cz14}, {"title": "CZ14 comparison"}]),
					dict(label="CZ15", method="update", args=[{"visible": plot_cz15}, {"title": "CZ15 comparison"}]),
					dict(label="CZ16", method="update", args=[{"visible": plot_cz16}, {"title": "CZ16 comparison"}]),   
							]),
				 )
					 ])

	fig5.update_layout(
		updatemenus=[
			dict(
				active=0,
				buttons=list([
					dict(label="All", method="update", args=[{"visible": [True]}, {"title": "All Peak day/weekday ratio"}]),
					dict(label="All set A", method="update", args=[{"visible": plot_A}, {"title": "All CZs, set A"}]),
					#dict(label="All set B", method="update", args=[{"visible": plot_B}, {"title": "All CZs, set B"}]),
					dict(label="CZ01", method="update", args=[{"visible": plot_cz1}, {"title": "CZ01 comparison"}]),
					dict(label="CZ02", method="update", args=[{"visible": plot_cz2}, {"title": "CZ02 comparison"}]),
					dict(label="CZ03", method="update", args=[{"visible": plot_cz3}, {"title": "CZ03 comparison"}]),
					dict(label="CZ04", method="update", args=[{"visible": plot_cz4}, {"title": "CZ04 comparison"}]),
					dict(label="CZ05", method="update", args=[{"visible": plot_cz5}, {"title": "CZ05 comparison"}]),
					dict(label="CZ06", method="update", args=[{"visible": plot_cz6}, {"title": "CZ06 comparison"}]),
					dict(label="CZ07", method="update", args=[{"visible": plot_cz7}, {"title": "CZ07 comparison"}]),
					dict(label="CZ08", method="update", args=[{"visible": plot_cz8}, {"title": "CZ08 comparison"}]),
					dict(label="CZ09", method="update", args=[{"visible": plot_cz9}, {"title": "CZ09 comparison"}]),
					dict(label="CZ10", method="update", args=[{"visible": plot_cz10}, {"title": "CZ10 comparison"}]),
					dict(label="CZ11", method="update", args=[{"visible": plot_cz11}, {"title": "CZ11 comparison"}]),
					dict(label="CZ12", method="update", args=[{"visible": plot_cz12}, {"title": "CZ12 comparison"}]),
					dict(label="CZ13", method="update", args=[{"visible": plot_cz13}, {"title": "CZ13 comparison"}]),
					dict(label="CZ14", method="update", args=[{"visible": plot_cz14}, {"title": "CZ14 comparison"}]),
					dict(label="CZ15", method="update", args=[{"visible": plot_cz15}, {"title": "CZ15 comparison"}]),
					dict(label="CZ16", method="update", args=[{"visible": plot_cz16}, {"title": "CZ16 comparison"}]),   
							]),
				 )
					 ])

	fig6.update_layout(
		updatemenus=[
			dict(
				active=0,
				buttons=list([
					dict(label="All", method="update", args=[{"visible": [True]}, {"title": "All weekend/weekday ratio"}]),
					dict(label="All set A", method="update", args=[{"visible": plot_A}, {"title": "All CZs, set A"}]),
					#dict(label="All set B", method="update", args=[{"visible": plot_B}, {"title": "All CZs, set B"}]),
					dict(label="CZ01", method="update", args=[{"visible": plot_cz1}, {"title": "CZ01 comparison"}]),
					dict(label="CZ02", method="update", args=[{"visible": plot_cz2}, {"title": "CZ02 comparison"}]),
					dict(label="CZ03", method="update", args=[{"visible": plot_cz3}, {"title": "CZ03 comparison"}]),
					dict(label="CZ04", method="update", args=[{"visible": plot_cz4}, {"title": "CZ04 comparison"}]),
					dict(label="CZ05", method="update", args=[{"visible": plot_cz5}, {"title": "CZ05 comparison"}]),
					dict(label="CZ06", method="update", args=[{"visible": plot_cz6}, {"title": "CZ06 comparison"}]),
					dict(label="CZ07", method="update", args=[{"visible": plot_cz7}, {"title": "CZ07 comparison"}]),
					dict(label="CZ08", method="update", args=[{"visible": plot_cz8}, {"title": "CZ08 comparison"}]),
					dict(label="CZ09", method="update", args=[{"visible": plot_cz9}, {"title": "CZ09 comparison"}]),
					dict(label="CZ10", method="update", args=[{"visible": plot_cz10}, {"title": "CZ10 comparison"}]),
					dict(label="CZ11", method="update", args=[{"visible": plot_cz11}, {"title": "CZ11 comparison"}]),
					dict(label="CZ12", method="update", args=[{"visible": plot_cz12}, {"title": "CZ12 comparison"}]),
					dict(label="CZ13", method="update", args=[{"visible": plot_cz13}, {"title": "CZ13 comparison"}]),
					dict(label="CZ14", method="update", args=[{"visible": plot_cz14}, {"title": "CZ14 comparison"}]),
					dict(label="CZ15", method="update", args=[{"visible": plot_cz15}, {"title": "CZ15 comparison"}]),
					dict(label="CZ16", method="update", args=[{"visible": plot_cz16}, {"title": "CZ16 comparison"}]),   
							]),
				 )
					 ])

	#fig4.show()
	#fig5.show()
	#fig6.show()
	return fig4, fig5, fig6
